#include "mpi.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
// map reduce
#define COORDINATOR 0
#define MAPPER1 1
#define MAPPER2 2
#define MAPPER3 3
#define REDUCER_VOWELS 4
#define REDUCER_CONSONANTS 5
#define MAX_LINE_LENGTH 16

int is_vowel(char c) {
    c = tolower(c);
    return (c == 'a' || c == 'e' || c == 'i' || c == 'o' || c == 'u');
}

int is_consonant(char c) {
    c = tolower(c);
    return (c >= 'a' && c <= 'z' && !is_vowel(c));
}

int main(int argc, char *argv[]) {
    int procs, rank;
    MPI_Init(&argc, &argv);
    MPI_Comm_size(MPI_COMM_WORLD, &procs);
    MPI_Comm_rank(MPI_COMM_WORLD, &rank);

    if (procs != 6) {
        if (rank == COORDINATOR) {
            fprintf(stderr, "This program requires exactly 6 tasks\n");
        }
        MPI_Finalize();
        exit(1);
    }

    if (argc < 2) {
        if (rank == COORDINATOR) {
            fprintf(stderr, "Usage: mpirun -np 6 %s \n", argv[0]);
        }
        MPI_Finalize();
        exit(1);
    }

    if (rank == COORDINATOR) {
        // Coordinator reads the file and distributes lines
        FILE *file = fopen(argv[1], "r");
        if (!file) {
            fprintf(stderr, "Error opening file: %s\n", argv[1]);
            MPI_Abort(MPI_COMM_WORLD, 1);
        }

        int N;
        fscanf(file, "%d\n", &N);

        char **lines = (char**)malloc(N * sizeof(char*));
        for (int i = 0; i < N; i++) {
            lines[i] = (char*)malloc(MAX_LINE_LENGTH * sizeof(char));
            fgets(lines[i], MAX_LINE_LENGTH, file);
            // Remove newline
            lines[i][strcspn(lines[i], "\n")] = 0;
        }
        fclose(file);

        // Distribute lines to mappers (round-robin for balance)
        int lines_per_mapper[3] = {0, 0, 0};
        for (int i = 0; i < N; i++) {
            int mapper_idx = i % 3;
            lines_per_mapper[mapper_idx]++;
        }

        // Send line counts to mappers
        for (int i = 0; i < 3; i++) {
            MPI_Send(&lines_per_mapper[i], 1, MPI_INT, MAPPER1 + i, 0, MPI_COMM_WORLD);
        }

        // Send lines to mappers
        int line_idx = 0;
        for (int mapper = 0; mapper < 3; mapper++) {
            for (int i = 0; i < lines_per_mapper[mapper]; i++) {
                MPI_Send(lines[line_idx], MAX_LINE_LENGTH, MPI_CHAR, MAPPER1 + mapper, 1, MPI_COMM_WORLD);
                line_idx++;
            }
        }

        // Free memory
        for (int i = 0; i < N; i++) {
            free(lines[i]);
        }
        free(lines);

    } else if (rank >= MAPPER1 && rank <= MAPPER3) {
        // Mapper: receive lines and count vowels/consonants
        int num_lines;
        MPI_Recv(&num_lines, 1, MPI_INT, COORDINATOR, 0, MPI_COMM_WORLD, MPI_STATUS_IGNORE);

        int vowel_count = 0;
        int consonant_count = 0;

        for (int i = 0; i < num_lines; i++) {
            char line[MAX_LINE_LENGTH];
            MPI_Recv(line, MAX_LINE_LENGTH, MPI_CHAR, COORDINATOR, 1, MPI_COMM_WORLD, MPI_STATUS_IGNORE);
            
            printf("[%d] %s\n", rank, line);
            fflush(stdout);

            for (int j = 0; line[j] != '\0'; j++) {
                if (is_vowel(line[j])) {
                    vowel_count++;
                } else if (is_consonant(line[j])) {
                    consonant_count++;
                }
            }
        }

        printf("[%d] v=%d c=%d\n", rank, vowel_count, consonant_count);
        fflush(stdout);

        // Send results to reducers
        MPI_Send(&vowel_count, 1, MPI_INT, REDUCER_VOWELS, 2, MPI_COMM_WORLD);
        MPI_Send(&consonant_count, 1, MPI_INT, REDUCER_CONSONANTS, 3, MPI_COMM_WORLD);

    } else if (rank == REDUCER_VOWELS) {
        // Reducer for vowels
        int total_vowels = 0;
        for (int mapper = MAPPER1; mapper <= MAPPER3; mapper++) {
            int vowel_count;
            MPI_Recv(&vowel_count, 1, MPI_INT, mapper, 2, MPI_COMM_WORLD, MPI_STATUS_IGNORE);
            total_vowels += vowel_count;
        }
        printf("[%d] Vowels: %d\n", rank, total_vowels);
        fflush(stdout);

    } else if (rank == REDUCER_CONSONANTS) {
        // Reducer for consonants
        int total_consonants = 0;
        for (int mapper = MAPPER1; mapper <= MAPPER3; mapper++) {
            int consonant_count;
            MPI_Recv(&consonant_count, 1, MPI_INT, mapper, 3, MPI_COMM_WORLD, MPI_STATUS_IGNORE);
            total_consonants += consonant_count;
        }
        printf("[%d] Consonants: %d\n", rank, total_consonants);
        fflush(stdout);
    }

    MPI_Finalize();
    return 0;
}